<?php


// Helper: Apply rounded corners to an image
function applyRoundedCorners($image, $radius)
{
    $width  = imagesx($image);
    $height = imagesy($image);

    $mask = imagecreatetruecolor($width, $height);
    imagealphablending($mask, false);
    imagesavealpha($mask, true);

    $transparent = imagecolorallocatealpha($mask, 0, 0, 0, 127);
    imagefill($mask, 0, 0, $transparent);

    $black = imagecolorallocate($mask, 0, 0, 0);

    imagefilledrectangle($mask, $radius, 0, $width - $radius, $height, $black);
    imagefilledrectangle($mask, 0, $radius, $width, $height - $radius, $black);

    imagefilledellipse($mask, $radius, $radius, $radius * 2, $radius * 2, $black);
    imagefilledellipse($mask, $width - $radius, $radius, $radius * 2, $radius * 2, $black);
    imagefilledellipse($mask, $radius, $height - $radius, $radius * 2, $radius * 2, $black);
    imagefilledellipse($mask, $width - $radius, $height - $radius, $radius * 2, $radius * 2, $black);

    for ($x = 0; $x < $width; $x++) {
        for ($y = 0; $y < $height; $y++) {
            if (imagecolorat($mask, $x, $y) == $transparent) {
                imagesetpixel($image, $x, $y, $transparent);
            }
        }
    }

    return $image;
}

// Helper: Draw bold text
function drawBoldText($image, $size, $angle, $x, $y, $color, $font, $text)
{
    $offset = 1;
    for ($i = -$offset; $i <= $offset; $i++) {
        for ($j = -$offset; $j <= $offset; $j++) {
            imagettftext($image, $size, $angle, $x + $i, $y + $j, $color, $font, $text);
        }
    }
}

// Helper: Draw centered bold text
function drawCenteredText($image, $text, $fontSize, $y, $color, $fontPath, $centerX)
{
    $bbox      = imagettfbbox($fontSize, 0, $fontPath, $text);
    $textWidth = $bbox[2] - $bbox[0];
    $textX     = $centerX - ($textWidth / 2);
    drawBoldText($image, $fontSize, 0, $textX, $y, $color, $fontPath, $text);
}

// Helper: Compress image to desired file size
function compressImageToFileSize($inputPath, $outputPath, $maxFileSizeKB)
{
    $quality = 75;
    $image   = imagecreatefromjpeg($inputPath);

    do {
        imagejpeg($image, $outputPath, $quality);
        $fileSizeKB = filesize($outputPath) / 1024;
        $quality -= 5;
    } while ($fileSizeKB > $maxFileSizeKB && $quality > 10);

    imagedestroy($image);
}

// Main function
function generatePoster($backgroundPath, $overlayPath, $outputPath, $fontPath, $details)
{
    extract($details); // Extracts: name, regNumber, panchayath, x, y, etc.

    $background = imagecreatefromstring(file_get_contents($backgroundPath));
    $overlay    = imagecreatefromstring(file_get_contents($overlayPath));

    if (!$background || !$overlay) {
        die('Error loading image(s).');
    }

    // Resize and round overlay image
    $resizedOverlay = imagecreatetruecolor($overlayWidth, $overlayHeight);
    imagealphablending($resizedOverlay, false);
    imagesavealpha($resizedOverlay, true);
    $transparent = imagecolorallocatealpha($resizedOverlay, 0, 0, 0, 127);
    imagefill($resizedOverlay, 0, 0, $transparent);
    imagecopyresampled($resizedOverlay, $overlay, 0, 0, 0, 0, $overlayWidth, $overlayHeight, imagesx($overlay), imagesy($overlay));
    $resizedOverlay = applyRoundedCorners($resizedOverlay, 80);

    // Place overlay
    imagecopy($background, $resizedOverlay, $x, $y, 0, 0, $overlayWidth, $overlayHeight);

    // Set text color
    $color = imagecolorallocate($background, ...$textColor);

    // Draw name, panchayath, and reg number
    drawCenteredText($background, $name, $fontSize, 2800, $color, $fontPath, 1850);
    drawCenteredText($background, $panchayath, $fontSize, 2950, $color, $fontPath, 1850);

    $regFontSize = 60;
    $regY        = $y + $overlayHeight + 120;
    $regCenterX  = $x + ($overlayWidth / 2);
    drawCenteredText($background, $regNumber, $regFontSize, $regY, $color, $fontPath, $regCenterX);

    // Save and compress
    $tempOutputPath = 'temp_output.jpg';
    imagejpeg($background, $tempOutputPath, 75);
    compressImageToFileSize($tempOutputPath, $outputPath, $maxFileSizeKB);

    // Output to browser
    header('Content-Type: image/jpeg');
    header('Content-Disposition: attachment; filename="poster.jpg"');
    readfile($outputPath);

    // Cleanup
    unlink($tempOutputPath);
    unlink($outputPath);
    imagedestroy($background);
    imagedestroy($overlay);
    imagedestroy($resizedOverlay);
}

// ==== Execution Starts Here ====
$name       = $_GET['name']       ?? '';
$regNumber  = $_GET['regno']      ?? '';
$panchayath = $_GET['panchayath'] ?? '';
$image      = $_GET['image']      ?? '';
$exam       = $_GET['exam']       ?? '10';

if (!$name || !$regNumber || !$image || !$panchayath) {
    die('Missing required parameters.');
}

$backgroundPath = ($exam == 10) ? '1.png' : '2.png';
$overlayPath    = $image;
$outputPath     = 'poster.jpg';
$fontPath       = 'Arial.ttf'; // Ensure this exists and is valid

// Parameters
$details = [
    'name'          => $name,
    'regNumber'     => $regNumber,
    'panchayath'    => $panchayath,
    'x'             => 250,
    'y'             => 2400,
    'overlayWidth'  => 739,
    'overlayHeight' => 881,
    'fontSize'      => 75,
    'textColor'     => [255, 255, 255], // white
    'maxFileSizeKB' => 500,
];

generatePoster($backgroundPath, $overlayPath, $outputPath, $fontPath, $details);
